# HDGLState to zk-SNARK input conversion for Solana-based data
from hdgl_operators import blend_slot, STRANDS, WAVES, SCALE
from web3 import Web3
from decimal import Decimal, getcontext
getcontext().prec = 50

def solana_to_zk_input(state: dict, r_dim_max=Decimal("1.0")) -> dict:
    # Convert scaled r_dim / omega
    r_dim = Decimal(state["r_dim_scaled"]) / Decimal(1_000_000_000)
    Omega = Decimal(state["omega_scaled"]) / Decimal(1_000_000_000)

    # Take first 8 slots for D_slots (or weighted aggregation)
    raw_slots = [Decimal(x)/Decimal(1_000_000_000) for x in state["lattice"][:8]]
    D_slots = [blend_slot(raw_slots, i, r_dim) for i in range(8)]

    # Commitment (Ethereum-style Keccak256)
    commitment_bytes = b''.join([x.to_bytes(32,'big') for x in D_slots] +
                                [int(r_dim*SCALE).to_bytes(32,'big'),
                                 int(Omega*SCALE).to_bytes(32,'big')])
    commitment = Web3.keccak(commitment_bytes).hex()

    return {
        "D_slots": [str(x) for x in D_slots],
        "WAVES": WAVES,
        "STRANDS": STRANDS,
        "r_dim_scaled": str(int(r_dim*SCALE)),
        "Omega_scaled": str(int(Omega*SCALE)),
        "commitment": commitment
    }
